<?php
/**
 * Plugin Name: 402gate
 * Version: 1.0.1
 * Description: Payment gateway integration for x402 protocol.
 */

if (!defined('ABSPATH')) {
    exit;
}

// Include admin page
require_once plugin_dir_path(__FILE__) . 'admin-page.php';

// Helper function to get API URL
function x402_get_api_url()
{
    $url = get_option('x402_api_url');
    if (empty($url)) {
        return 'https://api.402gate.xyz';
    }
    return untrailingslashit($url);
}

// Telemetry Helper
function x402_telemetry($event_type, $meta = array())
{
    $url = get_option('x402_telemetry_url', 'https://four02gate.onrender.com/collect');
    if (empty($url)) {
        return; // Telemetry not configured
    }

    $wallet = get_option('x402_wallet');
    $price = get_option('x402_price');

    $body = array(
        'event_type' => $event_type,
        'project_type' => 'wordpress-plugin',
        'wallet' => $wallet,
        'amount' => $price,
        'meta' => $meta
    );

    wp_remote_post($url, array(
        'blocking' => false, // Fire and forget
        'headers' => array('Content-Type' => 'application/json'),
        'body' => json_encode($body)
    ));
}

// Schedule Daily Heartbeat
register_activation_hook(__FILE__, 'x402_activate_cron');
function x402_activate_cron()
{
    if (!wp_next_scheduled('x402_daily_heartbeat_event')) {
        wp_schedule_event(time(), 'daily', 'x402_daily_heartbeat_event');
    }
}

add_action('x402_daily_heartbeat_event', 'x402_send_heartbeat');
function x402_send_heartbeat()
{
    x402_telemetry('plugin_heartbeat', array('version' => '1.0'));
}

// Intercept content hook
add_action('template_redirect', 'x402_intercept_content');

function x402_intercept_content()
{
    // 1. Detection: Single post AND (bot param OR User-Agent)
    if (!is_single()) {
        return;
    }

    $is_bot_param = isset($_GET['bot']) && $_GET['bot'] === 'true';

    $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';

    // --- SEO WHITELIST ---
    // Allow Googlebot (Search Indexing) to access content for free.
    // BUT block Google-Extended (AI Training) -> they must pay.
    if (stripos($user_agent, 'Googlebot') !== false && stripos($user_agent, 'Google-Extended') === false) {
        return; // Whitelist: Exit function, allow normal WP rendering
    }

    // --- LISTA BOTÓW I AGENTÓW ---
    $bot_agents = array(
        // Główne AI & LLM
        'GPTBot',           // OpenAI Crawler
        'ChatGPT-User',     // ChatGPT Browsing
        'Google-Extended',  // Google Gemini/Bard control
        'GoogleOther',      // Inne zasoby Google
        'anthropic-ai',     // Claude
        'ClaudeBot',        // Claude Crawler
        'PerplexityBot',    // Perplexity AI
        'FacebookBot',      // Meta AI
        'Applebot',         // Apple Intelligence/Siri
        'Amazonbot',        // Alexa/Amazon Q
        'cohere-ai',        // Cohere
        'Diffbot',          // Knowledge Graph AI

        // Narzędzia Deweloperskie (używane przez customowe Agenty)
        'curl',
        'Python',           // biblioteka requests, urllib
        'Postman',
        'Insomnia',
        'wget',
        'httpie',
        'axios',            // Node.js client
        'node-fetch',       // Node.js client
        'go-http-client',   // Go Lang
        'Java/',            // Java clients
        'aiohttp',          // Python async client

        // Ogólne Crawlery
        'Bytespider',       // ByteDance (TikTok)
        'CCBot',            // Common Crawl
        'magpie-crawler'
    );

    $is_bot_agent = false;
    foreach ($bot_agents as $agent) {
        if (stripos($user_agent, $agent) !== false) {
            $is_bot_agent = true;
            break;
        }
    }

    if (!$is_bot_param && !$is_bot_agent) {
        return; // To nie bot, wpuść człowieka za darmo
    }

    // 2. Retrieve Settings
    $wallet = get_option('x402_wallet');
    $price = get_option('x402_price');
    $api_url = x402_get_api_url();

    // Defaults / Fallbacks
    if (empty($wallet))
        $wallet = '0x505bc35f0a83c9ed06c6f94e68f0f86cf2812a6b';
    if (empty($price))
        $price = '0.01';

    // 3. Check for Payment Header
    // Sprawdzamy standardowy x402 oraz warianty
    $payment_hash = null;
    if (isset($_SERVER['HTTP_X_PAYMENT_HASH'])) {
        $payment_hash = $_SERVER['HTTP_X_PAYMENT_HASH'];
    } elseif (isset($_SERVER['HTTP_X_PAYMENT'])) { // Zgodność z whitepaperem
        $payment_hash = $_SERVER['HTTP_X_PAYMENT'];
    }

    // Scenario A: No Payment Hash -> Return 402 Offer
    if (!$payment_hash) {
        status_header(402);
        header('Content-Type: application/json');

        // JSON zgodny ze specyfikacją x402
        $response = array(
            'maxAmountRequired' => $price,
            'payTo' => $wallet,
            'resource' => get_permalink(),
            'asset' => '0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913', // USDC on Base
            'network' => 'base-mainnet',
            'description' => 'Premium content via 402gate'
        );

        echo json_encode($response);

        // Telemetry: Impression
        x402_telemetry('payment_required', array('url' => get_permalink()));

        exit; // Zatrzymaj ładowanie strony
    }

    // Scenario B: Verify Payment Hash with Brain (Backend)
    $verify_url = $api_url . '/x402/verify';

    $response = wp_remote_post($verify_url, array(
        'headers' => array('Content-Type' => 'application/json'),
        'body' => json_encode(array('paymentHash' => $payment_hash)),
        'timeout' => 15,
    ));

    if (is_wp_error($response)) {
        status_header(500);
        header('Content-Type: application/json');
        echo json_encode(array('error' => 'Internal Server Error: ' . $response->get_error_message()));
        exit;
    }

    $response_code = wp_remote_retrieve_response_code($response);

    if ($response_code === 200) {
        // SUKCES! Płatność potwierdzona.
        // Zwracamy towar premium: Czysty JSON bez HTML-owego śmietnika.

        header('Content-Type: application/json');

        $post = get_post();

        // Czyszczenie treści z tagów HTML
        $clean_content = strip_tags($post->post_content);

        echo json_encode(array(
            'status' => 'paid',
            'access' => 'granted',
            'title' => get_the_title(),
            'content' => $clean_content, // To jest to, za co bot zapłacił
            'id' => get_the_ID(),
            'date' => get_the_date('c')
        ));

        // Telemetry: Success
        x402_telemetry('payment_success', array('hash' => $payment_hash));

        exit; // Ważne: Zatrzymujemy WordPressa, żeby nie dokleił reszty strony HTML
    } else {
        // Payment Invalid
        status_header(402);
        header('Content-Type: application/json');
        echo json_encode(array('error' => 'Payment Invalid or Expired'));

        // Telemetry: Failed
        x402_telemetry('payment_failed', array('hash' => $payment_hash, 'code' => $response_code));

        exit;
    }
}